// Copyright 2000-2024 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.psi.elements;

import com.intellij.openapi.util.text.StringUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.Serializable;

public final class PhpModifier implements Serializable {

  public static final PhpModifier PRIVATE_ABSTRACT_DYNAMIC = new PhpModifier(Access.PRIVATE, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PRIVATE_ABSTRACT_STATIC = new PhpModifier(Access.PRIVATE, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PRIVATE_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PRIVATE, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PRIVATE_IMPLEMENTED_STATIC = new PhpModifier(Access.PRIVATE, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PRIVATE_FINAL_DYNAMIC = new PhpModifier(Access.PRIVATE, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PRIVATE_FINAL_STATIC = new PhpModifier(Access.PRIVATE, Abstractness.FINAL, State.STATIC);

  public static final PhpModifier PROTECTED_ABSTRACT_DYNAMIC = new PhpModifier(Access.PROTECTED, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PROTECTED_ABSTRACT_STATIC = new PhpModifier(Access.PROTECTED, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PROTECTED_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PROTECTED, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PROTECTED_IMPLEMENTED_STATIC = new PhpModifier(Access.PROTECTED, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PROTECTED_FINAL_DYNAMIC = new PhpModifier(Access.PROTECTED, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PROTECTED_FINAL_STATIC = new PhpModifier(Access.PROTECTED, Abstractness.FINAL, State.STATIC);

  public static final PhpModifier PUBLIC_ABSTRACT_DYNAMIC = new PhpModifier(Access.PUBLIC, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PUBLIC_ABSTRACT_STATIC = new PhpModifier(Access.PUBLIC, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PUBLIC_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PUBLIC, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PUBLIC_IMPLEMENTED_STATIC = new PhpModifier(Access.PUBLIC, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PUBLIC_FINAL_DYNAMIC = new PhpModifier(Access.PUBLIC, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PUBLIC_FINAL_STATIC = new PhpModifier(Access.PUBLIC, Abstractness.FINAL, State.STATIC);

  // 8.4 asymmetric visibility
  public static final PhpModifier PUBLIC_PRIVATE_ABSTRACT_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PRIVATE, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PROTECTED_ABSTRACT_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PROTECTED, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PUBLIC_ABSTRACT_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PUBLIC, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PRIVATE_ABSTRACT_STATIC = new PhpModifier(Access.PUBLIC, Access.PRIVATE, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PUBLIC_PROTECTED_ABSTRACT_STATIC = new PhpModifier(Access.PUBLIC, Access.PROTECTED, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PUBLIC_PUBLIC_ABSTRACT_STATIC = new PhpModifier(Access.PUBLIC, Access.PUBLIC, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PUBLIC_PRIVATE_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PRIVATE, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PROTECTED_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PROTECTED, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PUBLIC_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PUBLIC, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PRIVATE_IMPLEMENTED_STATIC = new PhpModifier(Access.PUBLIC, Access.PRIVATE, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PUBLIC_PROTECTED_IMPLEMENTED_STATIC = new PhpModifier(Access.PUBLIC, Access.PROTECTED, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PUBLIC_PUBLIC_IMPLEMENTED_STATIC = new PhpModifier(Access.PUBLIC, Access.PUBLIC, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PUBLIC_PRIVATE_FINAL_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PRIVATE, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PROTECTED_FINAL_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PROTECTED, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PUBLIC_FINAL_DYNAMIC = new PhpModifier(Access.PUBLIC, Access.PUBLIC, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PUBLIC_PRIVATE_FINAL_STATIC = new PhpModifier(Access.PUBLIC, Access.PRIVATE, Abstractness.FINAL, State.STATIC);
  public static final PhpModifier PUBLIC_PROTECTED_FINAL_STATIC = new PhpModifier(Access.PUBLIC, Access.PROTECTED, Abstractness.FINAL, State.STATIC);
  public static final PhpModifier PUBLIC_PUBLIC_FINAL_STATIC = new PhpModifier(Access.PUBLIC, Access.PUBLIC, Abstractness.FINAL, State.STATIC);

  public static final PhpModifier PROTECTED_PUBLIC_ABSTRACT_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PUBLIC, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PRIVATE_ABSTRACT_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PRIVATE, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PROTECTED_ABSTRACT_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PROTECTED, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PUBLIC_ABSTRACT_STATIC = new PhpModifier(Access.PROTECTED, Access.PUBLIC, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PROTECTED_PRIVATE_ABSTRACT_STATIC = new PhpModifier(Access.PROTECTED, Access.PRIVATE, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PROTECTED_PROTECTED_ABSTRACT_STATIC = new PhpModifier(Access.PROTECTED, Access.PROTECTED, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PROTECTED_PUBLIC_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PUBLIC, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PRIVATE_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PRIVATE, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PROTECTED_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PROTECTED, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PUBLIC_IMPLEMENTED_STATIC = new PhpModifier(Access.PROTECTED, Access.PUBLIC, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PROTECTED_PRIVATE_IMPLEMENTED_STATIC = new PhpModifier(Access.PROTECTED, Access.PRIVATE, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PROTECTED_PROTECTED_IMPLEMENTED_STATIC = new PhpModifier(Access.PROTECTED, Access.PROTECTED, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PROTECTED_PUBLIC_FINAL_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PUBLIC, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PRIVATE_FINAL_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PRIVATE, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PROTECTED_FINAL_DYNAMIC = new PhpModifier(Access.PROTECTED, Access.PROTECTED, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PROTECTED_PUBLIC_FINAL_STATIC = new PhpModifier(Access.PROTECTED, Access.PUBLIC, Abstractness.FINAL, State.STATIC);
  public static final PhpModifier PROTECTED_PRIVATE_FINAL_STATIC = new PhpModifier(Access.PROTECTED, Access.PRIVATE, Abstractness.FINAL, State.STATIC);
  public static final PhpModifier PROTECTED_PROTECTED_FINAL_STATIC = new PhpModifier(Access.PROTECTED, Access.PROTECTED, Abstractness.FINAL, State.STATIC);

  public static final PhpModifier PRIVATE_PUBLIC_ABSTRACT_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PUBLIC, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PROTECTED_ABSTRACT_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PROTECTED, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PRIVATE_ABSTRACT_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PRIVATE, Abstractness.ABSTRACT, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PUBLIC_ABSTRACT_STATIC = new PhpModifier(Access.PRIVATE, Access.PUBLIC, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PRIVATE_PROTECTED_ABSTRACT_STATIC = new PhpModifier(Access.PRIVATE, Access.PROTECTED, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PRIVATE_PRIVATE_ABSTRACT_STATIC = new PhpModifier(Access.PRIVATE, Access.PRIVATE, Abstractness.ABSTRACT, State.STATIC);
  public static final PhpModifier PRIVATE_PUBLIC_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PUBLIC, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PROTECTED_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PROTECTED, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PRIVATE_IMPLEMENTED_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PRIVATE, Abstractness.IMPLEMENTED, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PUBLIC_IMPLEMENTED_STATIC = new PhpModifier(Access.PRIVATE, Access.PUBLIC, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PRIVATE_PROTECTED_IMPLEMENTED_STATIC = new PhpModifier(Access.PRIVATE, Access.PROTECTED, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PRIVATE_PRIVATE_IMPLEMENTED_STATIC = new PhpModifier(Access.PRIVATE, Access.PRIVATE, Abstractness.IMPLEMENTED, State.STATIC);
  public static final PhpModifier PRIVATE_PUBLIC_FINAL_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PUBLIC, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PROTECTED_FINAL_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PROTECTED, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PRIVATE_FINAL_DYNAMIC = new PhpModifier(Access.PRIVATE, Access.PRIVATE, Abstractness.FINAL, State.DYNAMIC);
  public static final PhpModifier PRIVATE_PUBLIC_FINAL_STATIC = new PhpModifier(Access.PRIVATE, Access.PUBLIC, Abstractness.FINAL, State.STATIC);
  public static final PhpModifier PRIVATE_PROTECTED_FINAL_STATIC = new PhpModifier(Access.PRIVATE, Access.PROTECTED, Abstractness.FINAL, State.STATIC);
  public static final PhpModifier PRIVATE_PRIVATE_FINAL_STATIC = new PhpModifier(Access.PRIVATE, Access.PRIVATE, Abstractness.FINAL, State.STATIC);

  public static @NotNull PhpModifier instance(@NotNull Access access, @NotNull Abstractness abstractness, @NotNull State state) {
    return switch (state) {
      case DYNAMIC -> switch (access) {
        case PRIVATE -> switch (abstractness) {
          case ABSTRACT -> PRIVATE_ABSTRACT_DYNAMIC;
          case IMPLEMENTED -> PRIVATE_IMPLEMENTED_DYNAMIC;
          case FINAL -> PRIVATE_FINAL_DYNAMIC;
        };
        case PROTECTED -> switch (abstractness) {
          case ABSTRACT -> PROTECTED_ABSTRACT_DYNAMIC;
          case IMPLEMENTED -> PROTECTED_IMPLEMENTED_DYNAMIC;
          case FINAL -> PROTECTED_FINAL_DYNAMIC;
        };
        case PUBLIC -> switch (abstractness) {
          case ABSTRACT -> PUBLIC_ABSTRACT_DYNAMIC;
          case IMPLEMENTED -> PUBLIC_IMPLEMENTED_DYNAMIC;
          case FINAL -> PUBLIC_FINAL_DYNAMIC;
        };
      };
      case STATIC -> switch (access) {
        case PRIVATE -> switch (abstractness) {
          case ABSTRACT -> PRIVATE_ABSTRACT_STATIC;
          case IMPLEMENTED -> PRIVATE_IMPLEMENTED_STATIC;
          case FINAL -> PRIVATE_FINAL_STATIC;
        };
        case PROTECTED -> switch (abstractness) {
          case ABSTRACT -> PROTECTED_ABSTRACT_STATIC;
          case IMPLEMENTED -> PROTECTED_IMPLEMENTED_STATIC;
          case FINAL -> PROTECTED_FINAL_STATIC;
        };
        case PUBLIC -> switch (abstractness) {
          case ABSTRACT -> PUBLIC_ABSTRACT_STATIC;
          case IMPLEMENTED -> PUBLIC_IMPLEMENTED_STATIC;
          case FINAL -> PUBLIC_FINAL_STATIC;
        };
      };
      default -> throw new IllegalArgumentException("State = " + state + " is not expected");
    };
  }

  public static @NotNull PhpModifier instance(@NotNull Access access, @Nullable Access setAccess, @NotNull Abstractness abstractness, @NotNull State state) {
    if (setAccess == null) return instance(access, abstractness, state);
    return switch (state) {
        case DYNAMIC -> switch (access) {
          case PRIVATE -> switch (abstractness) {
            case ABSTRACT -> switch (setAccess) {
              case PRIVATE -> PRIVATE_PRIVATE_ABSTRACT_DYNAMIC;
              case PROTECTED -> PRIVATE_PROTECTED_ABSTRACT_DYNAMIC;
              case PUBLIC -> PRIVATE_PUBLIC_ABSTRACT_DYNAMIC;
            };
            case IMPLEMENTED -> switch (setAccess) {
              case PRIVATE -> PRIVATE_PRIVATE_IMPLEMENTED_DYNAMIC;
              case PROTECTED -> PRIVATE_PROTECTED_IMPLEMENTED_DYNAMIC;
              case PUBLIC -> PRIVATE_PUBLIC_IMPLEMENTED_DYNAMIC;
            };
            case FINAL -> switch (setAccess) {
              case PRIVATE -> PRIVATE_PRIVATE_FINAL_DYNAMIC;
              case PROTECTED -> PRIVATE_PROTECTED_FINAL_DYNAMIC;
              case PUBLIC -> PRIVATE_PUBLIC_FINAL_DYNAMIC;
            };
          };
          case PROTECTED -> switch (abstractness) {
            case ABSTRACT -> switch (setAccess) {
              case PRIVATE -> PROTECTED_PRIVATE_ABSTRACT_DYNAMIC;
              case PROTECTED -> PROTECTED_PROTECTED_ABSTRACT_DYNAMIC;
              case PUBLIC -> PROTECTED_PUBLIC_ABSTRACT_DYNAMIC;
            };
            case IMPLEMENTED -> switch (setAccess) {
              case PRIVATE -> PROTECTED_PRIVATE_IMPLEMENTED_DYNAMIC;
              case PROTECTED -> PROTECTED_PROTECTED_IMPLEMENTED_DYNAMIC;
              case PUBLIC -> PROTECTED_PUBLIC_IMPLEMENTED_DYNAMIC;
            };
            case FINAL -> switch (setAccess) {
              case PRIVATE -> PROTECTED_PRIVATE_FINAL_DYNAMIC;
              case PROTECTED -> PROTECTED_PROTECTED_FINAL_DYNAMIC;
              case PUBLIC -> PROTECTED_PUBLIC_FINAL_DYNAMIC;
            };
          };
          case PUBLIC -> switch (abstractness) {
            case ABSTRACT -> switch (setAccess) {
              case PRIVATE -> PUBLIC_PRIVATE_ABSTRACT_DYNAMIC;
              case PROTECTED -> PUBLIC_PROTECTED_ABSTRACT_DYNAMIC;
              case PUBLIC -> PUBLIC_PUBLIC_ABSTRACT_DYNAMIC;
            };
            case IMPLEMENTED -> switch (setAccess) {
              case PRIVATE -> PUBLIC_PRIVATE_IMPLEMENTED_DYNAMIC;
              case PROTECTED -> PUBLIC_PROTECTED_IMPLEMENTED_DYNAMIC;
              case PUBLIC -> PUBLIC_PUBLIC_IMPLEMENTED_DYNAMIC;
            };
            case FINAL -> switch (setAccess) {
              case PRIVATE -> PUBLIC_PRIVATE_FINAL_DYNAMIC;
              case PROTECTED -> PUBLIC_PROTECTED_FINAL_DYNAMIC;
              case PUBLIC -> PUBLIC_PUBLIC_FINAL_DYNAMIC;
            };
          };
        };
        case STATIC -> switch (access) {
          case PRIVATE -> switch (abstractness) {
            case ABSTRACT -> switch (setAccess) {
              case PRIVATE -> PRIVATE_PRIVATE_ABSTRACT_STATIC;
              case PROTECTED -> PRIVATE_PROTECTED_ABSTRACT_STATIC;
              case PUBLIC -> PRIVATE_PUBLIC_ABSTRACT_STATIC;
            };
            case IMPLEMENTED -> switch (setAccess) {
              case PRIVATE -> PRIVATE_PRIVATE_IMPLEMENTED_STATIC;
              case PROTECTED -> PRIVATE_PROTECTED_IMPLEMENTED_STATIC;
              case PUBLIC -> PRIVATE_PUBLIC_IMPLEMENTED_STATIC;
            };
            case FINAL -> switch (setAccess) {
              case PRIVATE -> PRIVATE_PRIVATE_FINAL_STATIC;
              case PROTECTED -> PRIVATE_PROTECTED_FINAL_STATIC;
              case PUBLIC -> PRIVATE_PUBLIC_FINAL_STATIC;
            };
          };
          case PROTECTED -> switch (abstractness) {
            case ABSTRACT -> switch (setAccess) {
              case PRIVATE -> PROTECTED_PRIVATE_ABSTRACT_STATIC;
              case PROTECTED -> PROTECTED_PROTECTED_ABSTRACT_STATIC;
              case PUBLIC -> PROTECTED_PUBLIC_ABSTRACT_STATIC;
            };
            case IMPLEMENTED -> switch (setAccess) {
              case PRIVATE -> PROTECTED_PRIVATE_IMPLEMENTED_STATIC;
              case PROTECTED -> PROTECTED_PROTECTED_IMPLEMENTED_STATIC;
              case PUBLIC -> PROTECTED_PUBLIC_IMPLEMENTED_STATIC;
            };
            case FINAL -> switch (setAccess) {
              case PRIVATE -> PROTECTED_PRIVATE_FINAL_STATIC;
              case PROTECTED -> PROTECTED_PROTECTED_FINAL_STATIC;
              case PUBLIC -> PROTECTED_PUBLIC_FINAL_STATIC;
            };
          };
          case PUBLIC -> switch (abstractness) {
            case ABSTRACT -> switch (setAccess) {
              case PRIVATE -> PUBLIC_PRIVATE_ABSTRACT_STATIC;
              case PROTECTED -> PUBLIC_PROTECTED_ABSTRACT_STATIC;
              case PUBLIC -> PUBLIC_PUBLIC_ABSTRACT_STATIC;
            };
            case IMPLEMENTED -> switch (setAccess) {
              case PRIVATE -> PUBLIC_PRIVATE_IMPLEMENTED_STATIC;
              case PROTECTED -> PUBLIC_PROTECTED_IMPLEMENTED_STATIC;
              case PUBLIC -> PUBLIC_PUBLIC_IMPLEMENTED_STATIC;
            };
            case FINAL -> switch (setAccess) {
              case PRIVATE -> PUBLIC_PRIVATE_FINAL_STATIC;
              case PROTECTED -> PUBLIC_PROTECTED_FINAL_STATIC;
              case PUBLIC -> PUBLIC_PUBLIC_FINAL_STATIC;
            };
          };
        };
        default -> throw new IllegalArgumentException("State = " + state + " is not expected");
      };
  }

  public enum Access {
    PUBLIC(3), PROTECTED(2), PRIVATE(1);
    private final int myLevel;

    Access(int level) {
      myLevel = level;
    }

    public int getLevel() {
      return myLevel;
    }

    @Override
    public String toString() {
      return StringUtil.toLowerCase(super.toString());
    }

    public boolean isProtected() {
      return this == PROTECTED;
    }

    public boolean isPrivate() {
      return this == PRIVATE;
    }

    public boolean isPublic() {
      return this == PUBLIC;
    }

    public boolean isWeakerThan(Access access){
      return getLevelDiff(access) > 0;
    }

    public boolean isEqualOrWeakerThan(Access access){
      return getLevelDiff(access) >= 0;
    }

    private int getLevelDiff(Access access) {
      return this.myLevel - access.getLevel();
    }
  }

  public enum State {
    STATIC, PARENT, DYNAMIC;

    public boolean isStatic() {
      return this == STATIC;
    }
    public boolean isDynamic() {
      return this == DYNAMIC;
    }
  }

  public enum Abstractness {
    ABSTRACT, IMPLEMENTED, FINAL
  }

  private final Access access;
  private final Access setAccess;
  private final State state;
  private final Abstractness abstractness;

  private PhpModifier(@NotNull Access access, @Nullable Access setAccess, @NotNull Abstractness abstractness, @NotNull State state) {
    this.access = access;
    this.setAccess = setAccess;
    this.abstractness = abstractness;
    this.state = state;
  }

  private PhpModifier(@NotNull Access access, @NotNull Abstractness abstractness, @NotNull State state) {
    this.access = access;
    this.setAccess = null;
    this.abstractness = abstractness;
    this.state = state;
  }

  public @NotNull PhpModifier copy(@NotNull Access newAccess) {
    return new PhpModifier(newAccess, setAccess, abstractness, state);
  }

  public @NotNull PhpModifier copy(@NotNull Abstractness newAbstractness) {
    return new PhpModifier(access, setAccess, newAbstractness, state);
  }

  public @NotNull PhpModifier copy(@NotNull State newState) {
    return new PhpModifier(access, setAccess, abstractness, newState);
  }

  public @NotNull Abstractness getAbstractness() {
    return abstractness;
  }

  public @NotNull Access getAccess() {
    return access;
  }

  public @Nullable Access getSetAccess() {
    return setAccess;
  }

  public boolean isPublic() {
    return access == Access.PUBLIC;
  }

  public boolean isProtected() {
    return access == Access.PROTECTED;
  }

  public boolean isPrivate() {
    return access == Access.PRIVATE;
  }

  public boolean isSetPublic() {
    return setAccess == Access.PUBLIC;
  }

  public boolean isSetProtected() {
    return setAccess == Access.PROTECTED;
  }

  public boolean isSetPrivate() {
    return setAccess == Access.PRIVATE;
  }

  public @NotNull State getState() {
    return state;
  }

  public boolean isStatic() {
    return state == State.STATIC;
  }

  public boolean isDynamic() {
    return state == State.DYNAMIC;
  }

  public boolean isFinal() {
    return abstractness == Abstractness.FINAL;
  }

  public boolean isAbstract() {
    return abstractness == Abstractness.ABSTRACT;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder();
    if (isAbstract()) {
      buf.append("abstract ");
    }
    if (isPrivate()) {
      buf.append("private ");
    } else if (isProtected()) {
      buf.append("protected ");
    } else if (isPublic()) {
      buf.append("public ");
    }
    if (isSetPrivate()) {
      buf.append("private(set) ");
    } else if (isSetProtected()) {
      buf.append("protected(set) ");
    } else if (isSetPublic()) {
      buf.append("public(set) ");
    }
    if (isStatic()) {
      buf.append("static ");
    }
    if (isFinal()) {
      buf.append("final ");
    }
    return buf.toString().trim();
  }
}
