// Copyright 2000-2024 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.lang.psi.elements;

import com.intellij.openapi.util.Condition;
import com.intellij.psi.PsiElement;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.JBIterable;
import com.jetbrains.php.PhpIndex;
import com.jetbrains.php.lang.psi.resolve.types.PhpType;
import com.jetbrains.php.lang.psi.stubs.PhpClassStub;
import com.jetbrains.php.lang.psi.stubs.PhpStubBasedPsiElement;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Unmodifiable;

import javax.swing.*;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.intellij.util.containers.ContainerUtil.emptyList;

public interface PhpClass extends PhpNamedElement, PhpElementWithModifier, PhpStubBasedPsiElement<PhpClassStub>, PhpCallbackElement, PhpAttributesOwner {

  String CLONE = "__clone";
  String TO_STRING = "__toString";
  String CONSTRUCTOR = "__construct";
  String DESTRUCTOR = "__destruct";
  String INVOKE = "__invoke";
  String PARENT = "parent";
  String SELF = "self";
  String STATIC = "static";
  String CLASS = "class";
  PhpClass[] EMPTY_ARRAY = new PhpClass[0];
  Condition<PsiElement> INSTANCEOF = use -> use instanceof PhpClass;
  String ANONYMOUS = "__anonymous@";

  boolean isAnonymous();
  boolean isInterface();
  boolean isEnum();
  @NotNull PhpType getBackedEnumType();
  boolean isAbstract();
  boolean isFinal();
  boolean isReadonly();

  @NotNull ExtendsList getExtendsList();
  @NotNull ImplementsList getImplementsList();

  @Nullable
  String getSuperName();
  @Nullable
  String getSuperFQN();

  /**
   * Discuoraged, please consider to use {@link #getSuperClasses()} to correctly handle multi-resolve
   */
  @Nullable
  PhpClass getSuperClass();

  /**
   * Can return multiple classes for non-interface in case of multi-resolve
   */
  Collection<PhpClass> getSuperClasses();
  String @NotNull [] getInterfaceNames();
  PhpClass[] getImplementedInterfaces();

  @ApiStatus.Internal
  default @NotNull List<PhpClass> getDirectImplementedInterfaces() {
    final List<PhpClass> interfaceList = new ArrayList<>();
    final PhpIndex phpIndex = PhpIndex.getInstance(getProject());
    for (String fqn : getInterfaceNames()) {
      interfaceList.addAll(phpIndex.getInterfacesByFQN(fqn));
    }
    return interfaceList;
  }

  @NotNull @ApiStatus.Internal
  Collection<PhpClass> getImplicitEnumInterfaces();

  boolean hasTraitUses();
  String @NotNull [] getTraitNames();
  PhpClass[] getTraits();

  String[] getMixinNames();
  PhpClass[] getMixins();

  @ApiStatus.Internal
  boolean hasGenericMixins();
  @ApiStatus.Internal
  PhpClass[] getMixinsIncludingGeneric(String genericInstantiationType);

  PhpClass[] getSupers();

  /**
   * Heavy, looks across hierarchy. Try getOwnFields() & PhpClassHierarchyUtils.processFields(), findFieldByName
   */
  Collection<Field> getFields();
  /**
   * Try PhpClassHierarchyUtils.processFields(), findFieldByName
   */
  Field[] getOwnFields();

  @ApiStatus.Internal
  Field[] getOwnFields(boolean dynamicFieldsOnlyFromConstructor);
  /**
   * Heavy, looks across hierarchy. Result order unpredicted.
   * Try getOwnMethods(), PhpClassHierarchyUtils.processMethods(), findMethodByName
   */
  Collection<Method> getMethods();
  /**
   * Try PhpClassHierarchyUtils.processMethods(), findMethodByName
   *
   * @return only methods from the current class without traversing supers
   */
  Method[] getOwnMethods();

  boolean hasOwnStaticMembers();
  boolean hasStaticMembers();

  @Nullable
  Method getConstructor();
  @Nullable
  Method findMethodByName(@Nullable CharSequence name);

  @Nullable @ApiStatus.Internal
  Method findMethodByName(@Nullable CharSequence name, Collection<PhpClass> visited);

  @NotNull
  @ApiStatus.Internal
  Collection<Method> findMethodsByName(@Nullable CharSequence name);

  @NotNull
  @ApiStatus.Internal
  @Unmodifiable
  Collection<Method> findMethodsByName(@Nullable CharSequence name, HashSet<PhpClass> visited);
  @Nullable
  Method findOwnMethodByName(@Nullable CharSequence name);
  @Nullable
  Field findFieldByName(@Nullable CharSequence name, boolean findConstant);

  @ApiStatus.Internal
  @NotNull Collection<PhpClass> getTypeAwareImplicitEnumInterfaces();

  @Nullable
  Field findOwnFieldByName(@Nullable CharSequence name, boolean findConstant);
  @ApiStatus.Internal
  PhpOverloadedMethods findOwnMethodsByName(@Nullable CharSequence name);

  @Override
  @NotNull
  Icon getIcon();

  boolean hasMethodTags();
  boolean hasPropertyTags();
  boolean hasConstructorFields();

  @Nullable
  Method getOwnConstructor();

  boolean isTrait();

  JBIterable<PhpTraitUseRule> traitUseRules();

  @NotNull
  String getPresentableFQN();

  Collection<PhpEnumCase> getEnumCases();

  @ApiStatus.Internal
  enum PhpDynamicFieldInitPlace {
    CONSTRUCTOR, PHPUNIT_SETUP, PHPUNIT_SETUP_BEFORE_CLASS, DYNAMIC
  }

  @ApiStatus.Internal
  class PhpOverloadedMethods {
    public static final PhpOverloadedMethods EMPTY = new PhpOverloadedMethods(emptyList());
    private final Collection<? extends Method> myMethods;

    private final boolean myFromMixin;

    private PhpOverloadedMethods(Collection<? extends Method> methods) {
      this(methods, false);
    }

    private PhpOverloadedMethods(Collection<? extends Method> methods, boolean mixin) {
      myMethods = methods;
      myFromMixin = mixin;
    }

    public @Unmodifiable @NotNull Collection<Method> getMethods() {
      return Collections.unmodifiableCollection(myMethods);
    }

    public @Nullable Method getAny() {
      return ContainerUtil.getFirstItem(myMethods);
    }

    public boolean isEmpty() {
      return myMethods.isEmpty();
    }

    public @NotNull PhpOverloadedMethods filter(@NotNull Predicate<? super Method> predicate) {
      return from(ContainerUtil.filter(myMethods, predicate::test)).fromMixin(isFromMixin());
    }

    public boolean isFromMixin() {
      return myFromMixin;
    }

    public static @NotNull PhpOverloadedMethods from(Collection<? extends Method> methods) {
      return methods.isEmpty() ? EMPTY : new PhpOverloadedMethods(methods);
    }

    public static PhpOverloadedMethods fromMethods(Collection<PhpOverloadedMethods> values) {
      if (values.size() == 1) {
        return ContainerUtil.getFirstItem(values);
      }
      PhpOverloadedMethods res = from(values.stream().flatMap(m -> m.getMethods().stream()).collect(Collectors.toSet()));
      return updateWithMixin(values, res);
    }

    private static PhpOverloadedMethods updateWithMixin(Collection<PhpOverloadedMethods> values, PhpOverloadedMethods res) {
      return res.fromMixin(!values.isEmpty() && ContainerUtil.all(values, PhpOverloadedMethods::isFromMixin));
    }

    public @NotNull PhpOverloadedMethods fromMixin(boolean value) {
      return new PhpOverloadedMethods(myMethods, value);
    }
  }
}
