// Copyright 2000-2024 JetBrains s.r.o. and contributors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.jetbrains.php.codeInsight.controlFlow.instructions;

import com.jetbrains.php.lang.psi.elements.PhpPsiElement;
import com.jetbrains.php.lang.psi.elements.RWAccess;
import org.jetbrains.annotations.NotNull;

public interface PhpAccessInstruction extends PhpInstruction {

  @Override
  @NotNull
  PhpPsiElement getAnchor();

  @NotNull
  Access getAccess();

  interface Access{

    boolean isRead();

    boolean isWrite();

    boolean isWriteRef();

    boolean isReadRef();

    boolean isReadWrite();

    boolean isUnset();

    boolean isLightRead();

    Access WRITE_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {
      @Override
      public boolean isWrite() {
        return true;
      }

      @Override
      public String toString() {
        return "WRITE";
      }
    };
    Access WRITE_REF_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {
      @Override
      public boolean isWriteRef() {
        return true;
      }

      @Override
      public String toString() {
        return "WRITE_REF";
      }
    };
    Access READ_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {
      @Override
      public boolean isRead() {
        return true;
      }

      @Override
      public String toString() {
        return "READ";
      }
    };
    Access READ_REF_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {
      @Override
      public boolean isReadRef() {
        return true;
      }

      @Override
      public String toString() {
        return "READ_REF";
      }
    };
    Access READ_WRITE_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {
      @Override
      public boolean isRead() {
        return true;
      }

      @Override
      public boolean isWrite() {
        return true;
      }

      @Override
      public boolean isReadWrite() {
        return true;
      }

      @Override
      public String toString() {
        return "READ_WRITE";
      }
    };

    Access COMPACT_READ_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {
      @Override
      public boolean isLightRead() {
        return true;
      }

      @Override
      public String toString() {
        return "COMPACT_STRING_READ";
      }
    };


    Access LIGHT_READ_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {
      @Override
      public boolean isLightRead() {
        return true;
      }

      @Override
      public String toString() {
        return "LIGHT_READ";
      }
    };
    //light read accesses
    Access UNSET_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {
      @Override
      public boolean isUnset() {
        return true;
      }

      @Override
      public boolean isLightRead() {
        return true;
      }

      @Override
      public String toString() {
        return "UNSET";
      }
    };

    Access ISSET_ACCESS = new PhpAccessVariableInstruction.Access.DefaultAccess() {

      @Override
      public boolean isLightRead() {
        return true;
      }

      @Override
      public String toString() {
        return "ISSET";
      }
    };

    class DefaultAccess implements Access {
      @Override
      public boolean isRead() {
        return false;
      }

      @Override
      public boolean isWrite() {
        return false;
      }

      @Override
      public boolean isWriteRef() {
        return false;
      }

      @Override
      public boolean isReadRef() {
        return false;
      }

      @Override
      public boolean isReadWrite() {
        return false;
      }

      @Override
      public boolean isUnset() {
        return false;
      }

      @Override
      public boolean isLightRead() {
        return false;
      }
    }
  }

  class ReadOrReadRefAccess extends Access.DefaultAccess {

    private final RWAccess myVariable;

    public ReadOrReadRefAccess(final @NotNull RWAccess variable) {
      myVariable = variable;
    }

    @Override
    public boolean isReadRef() {
      return myVariable.isWriteAccess();
    }

    @Override
    public boolean isRead() {
      return !isReadRef();
    }

    @Override
    public String toString() {
      return "READ_OR_READ_REF";
    }
  }
}
